use std::borrow::Cow;

use rustc_errors::Applicability;
use rustc_hir::{Expr, ExprKind, Mutability, TyKind};
use rustc_lint::LateContext;
use rustc_middle::ty::{self, TypeAndMut};
use rustc_semver::RustcVersion;

use if_chain::if_chain;

use crate::utils::sugg::Sugg;
use crate::utils::{meets_msrv, span_lint_and_sugg};

use super::PTR_AS_PTR;

const PTR_AS_PTR_MSRV: RustcVersion = RustcVersion::new(1, 38, 0);

pub(super) fn check(cx: &LateContext<'tcx>, expr: &'tcx Expr<'_>, msrv: &Option<RustcVersion>) {
    if !meets_msrv(msrv.as_ref(), &PTR_AS_PTR_MSRV) {
        return;
    }

    if_chain! {
        if let ExprKind::Cast(cast_expr, cast_to_hir_ty) = expr.kind;
        let (cast_from, cast_to) = (cx.typeck_results().expr_ty(cast_expr), cx.typeck_results().expr_ty(expr));
        if let ty::RawPtr(TypeAndMut { mutbl: from_mutbl, .. }) = cast_from.kind();
        if let ty::RawPtr(TypeAndMut { ty: to_pointee_ty, mutbl: to_mutbl }) = cast_to.kind();
        if matches!((from_mutbl, to_mutbl),
            (Mutability::Not, Mutability::Not) | (Mutability::Mut, Mutability::Mut));
        // The `U` in `pointer::cast` have to be `Sized`
        // as explained here: https://github.com/rust-lang/rust/issues/60602.
        if to_pointee_ty.is_sized(cx.tcx.at(expr.span), cx.param_env);
        then {
            let mut applicability = Applicability::MachineApplicable;
            let cast_expr_sugg = Sugg::hir_with_applicability(cx, cast_expr, "_", &mut applicability);
            let turbofish = match &cast_to_hir_ty.kind {
                    TyKind::Infer => Cow::Borrowed(""),
                    TyKind::Ptr(mut_ty) if matches!(mut_ty.ty.kind, TyKind::Infer) => Cow::Borrowed(""),
                    _ => Cow::Owned(format!("::<{}>", to_pointee_ty)),
                };
            span_lint_and_sugg(
                cx,
                PTR_AS_PTR,
                expr.span,
                "`as` casting between raw pointers without changing its mutability",
                "try `pointer::cast`, a safer alternative",
                format!("{}.cast{}()", cast_expr_sugg.maybe_par(), turbofish),
                applicability,
            );
        }
    }
}
