use crate::command_prelude::*;
use anyhow::{anyhow, Context as _};
use cargo::core::compiler::future_incompat::{OnDiskReport, FUTURE_INCOMPAT_FILE};
use cargo::drop_eprint;
use std::io::Read;

pub fn cli() -> App {
    subcommand("report")
        .about("Generate and display various kinds of reports")
        .after_help("Run `cargo help report` for more detailed information.\n")
        .setting(clap::AppSettings::SubcommandRequiredElseHelp)
        .subcommand(
            subcommand("future-incompatibilities")
                .about("Reports any crates which will eventually stop compiling")
                .arg(
                    opt(
                        "id",
                        "identifier of the report generated by a Cargo command invocation",
                    )
                    .value_name("id")
                    .required(true),
                ),
        )
}

pub fn exec(config: &mut Config, args: &ArgMatches<'_>) -> CliResult {
    if !config.nightly_features_allowed {
        return Err(anyhow!("`cargo report` can only be used on the nightly channel").into());
    }
    match args.subcommand() {
        ("future-incompatibilities", Some(args)) => report_future_incompatibilies(config, args),
        (cmd, _) => panic!("unexpected command `{}`", cmd),
    }
}

fn report_future_incompatibilies(config: &Config, args: &ArgMatches<'_>) -> CliResult {
    let ws = args.workspace(config)?;
    let report_file = ws.target_dir().open_ro(
        FUTURE_INCOMPAT_FILE,
        ws.config(),
        "Future incompatible report",
    )?;

    let mut file_contents = String::new();
    report_file
        .file()
        .read_to_string(&mut file_contents)
        .with_context(|| "failed to read report")?;
    let on_disk_report: OnDiskReport =
        serde_json::from_str(&file_contents).with_context(|| "failed to load report")?;

    let id = args.value_of("id").unwrap();
    if id != on_disk_report.id {
        return Err(anyhow!(
            "Expected an id of `{}`, but `{}` was provided on the command line. \
             Your report may have been overwritten by a different one.",
            on_disk_report.id,
            id
        )
        .into());
    }

    drop_eprint!(config, "{}", on_disk_report.report);
    Ok(())
}
